#include "StdV3D.h"
#include "BlobBase.h"
#include "Triangulator.h"
#include "IsoSurfaceBlobsObject.h"




namespace V3D {

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// Constructeur / Destructeur ....

IsoSurfaceBlobsObject::IsoSurfaceBlobsObject(float fEnergyThresh)
	: m_fEnergyThreshold( fEnergyThresh )
{}

                           /////////////////////////////

IsoSurfaceBlobsObject::IsoSurfaceBlobsObject(const BlobPtrArray& apBlobs, float fEnergyThresh)
	: m_fEnergyThreshold(fEnergyThresh),
	  m_apBlobs(apBlobs)
{}

                           /////////////////////////////

IsoSurfaceBlobsObject::~IsoSurfaceBlobsObject()
{}

//////////////////////////////////////////////////////////////////////////////
// Accesseurs et derives


void IsoSurfaceBlobsObject::SetEnergyThresh( float fThresh)
{
	m_fEnergyThreshold = fThresh;
}

                           /////////////////////////////


// Ajoute un blob a la liste des blobs. (par reference car on veut s'assurer que le pointeur n'est pas null)
int32 IsoSurfaceBlobsObject::AddBlob( const BlobBase& blob)
{
	m_apBlobs.push_back( &blob);
	return m_apBlobs.size()-1;
}

                           /////////////////////////////

// Suppression de toutes les references aux blobs de l'isosurface
void IsoSurfaceBlobsObject::RemoveAllBlobs()
{
	m_apBlobs.resize(0);
}

//////////////////////////////////////////////////////////////////////////////


// Calcule les energies en bloc sur une grille plane (Z constant)
void IsoSurfaceBlobsObject::FuncValuesForPlaneGrid( float* pafValues, 
                                          const Vector3D& vcStart, LengthType rDeltaX, LengthType rDeltaY,
                                          int32 nPointsInLayerX, int32 nPointsInLayerY,
                                          const AllScansArray* paScansCubesToVisit /* = 0 */) const
{
	ClearPlaneGridValues( pafValues, nPointsInLayerX, nPointsInLayerY, paScansCubesToVisit);

	Vector3D vcSteps( rDeltaX, rDeltaY, 1.f);
	// On evalue la somme des energies recues par chaque blob au point 'pos'
	LengthType rPlaneZ = vcStart.z;

	int32 nBlobsCount = m_apBlobs.size();
	for( int32 i = 0; i < nBlobsCount; ++i)
	{
		const BlobBase* pCurBlob = m_apBlobs[i];
		const Vector3D& vcBlobBoundMin = pCurBlob->GetBlobBoundsMin();
		const Vector3D& vcBlobBoundMax = pCurBlob->GetBlobBoundsMax();
		if( rPlaneZ >= vcBlobBoundMin.z && rPlaneZ <= vcBlobBoundMax.z )
			pCurBlob->EnergyAddAtGrid(pafValues, vcStart, vcSteps, nPointsInLayerX, nPointsInLayerY, 1);
	}
}


                           /////////////////////////////

// Initialisation des valeurs d'energie pour la grille
//   si on connait les cubes interessants, on ne nettoie que les energies correspondantes
void IsoSurfaceBlobsObject::ClearPlaneGridValues( float* pafValues, 
                                          int32 nPointsInLayerX, int32 nPointsInLayerY,
                                          const AllScansArray* paScansCubesToClean /* = 0 */) const
{
	const float fEnergyInit = -m_fEnergyThreshold; 

	if( paScansCubesToClean != 0)
	{
		const AllScansArray& aScansCubesToClean = *paScansCubesToClean;

		const OneLineScansArray& slLinesScans = aScansCubesToClean.GetLinesScans();
		int32 nLinesRangesCount = slLinesScans.GetNbScans();

		for( int32 idxLineScan = 0; idxLineScan < nLinesRangesCount; ++idxLineScan)
		{
			const ScanLineRange& curLinesRange = slLinesScans[idxLineScan];
			int32 nBegY = curLinesRange.nRangeBeg;
			int32 nEndY = curLinesRange.nRangeEnd;

			for( int32 j = nBegY; j <= nEndY; ++j)
			{
				const OneLineScansArray& curLine = aScansCubesToClean[j];
				int32 nCurLineScansCount = curLine.GetNbScans();
				// Toute ligne visitee doit contenir au moins un scan
				assert(  nCurLineScansCount != 0);

				int32 nStartPointIndexLine = j * nPointsInLayerX;

				// On Parcourt les scans de cubes a visiter
				for( int32 iScan = 0; iScan < nCurLineScansCount; ++iScan)
				{
					const ScanLineRange& curScan = curLine[iScan];
					int32 nBegScanPoint = nStartPointIndexLine + curScan.nRangeBeg;
					int32 nEndScanPoint = nStartPointIndexLine + curScan.nRangeEnd + 1;
					assert( curScan.nRangeEnd  + 1 < nPointsInLayerX);
					// Pour chaque scan on visite tous les cubes
					for( int32 i = nBegScanPoint; i <= nEndScanPoint; ++i)
					{
						pafValues[i] = fEnergyInit;
						pafValues[i+nPointsInLayerX] = fEnergyInit;
					}
				}
			}
		}
	}
	else
	{
		// Si on ne dispose pas du sous ensemble des cubes a nettoyer, on nettoie tout
		// ATTENTION : goulot d'etranglement, lorsque la grille est grande!
		int32 nPositions = nPointsInLayerX * nPointsInLayerY;
		for( int32 i = 0; i < nPositions; ++i)
			pafValues[i] = fEnergyInit;
	}
}



                           /////////////////////////////

// Calcule l'energie en un point en tenant compte de tous les blobs
// Attention : pas optimal, il vaut mieux passer par les methodes qui calculent
// les valeurs sur une grille
float IsoSurfaceBlobsObject::FuncValueAtPos( const Vector3D& pos) const
{
	int32 nBlobsCount = m_apBlobs.size();

	// On evalue la somme des energies recues par chaque blob au point 'pos'
	float fFuncVal = -m_fEnergyThreshold;
	for( int32 i = 0; i < nBlobsCount; ++i)
	{
		const BlobBase* pCurBlob = m_apBlobs[i];
		pCurBlob->EnergyAddAtPos(fFuncVal, pos);
	}
	return fFuncVal;
}

                           /////////////////////////////



void IsoSurfaceBlobsObject::AddSurfaceNormalsFindInfluenced( Vect3fArray& avNorms, BlobsInfluencesIndices& influenced,
                                                             const Vect3DArray& aPos,
	                                                         const LocalisationInfoArray* paVtxLocalisation /* = 0 */) const
{
	AddSurfaceNormalsFindInfluencedSubArray( avNorms, influenced, aPos, 0, aPos.size(), paVtxLocalisation);
}

                           /////////////////////////////


void IsoSurfaceBlobsObject::AddSurfaceNormalsFindInfluencedSubArray( Vect3fArray& avNorms, BlobsInfluencesIndices& influenced,
                                                                     const Vect3DArray& aPos,
                                                                     int32 nStartIdx, int32 nCount,
	                                                                 const LocalisationInfoArray* paVtxLocalisation /* = 0 */) const
{
	int32 nBlobsCount = m_apBlobs.size();
	influenced.resize( nBlobsCount );
		
//	int32 nVtxCount = aPos.size();
//	int32 nEvalInfluencedPerBlob = ( 50 * nVtxCount) / nBlobsCount;
	int32 nEvalInfluencedPerBlob = int32( (  3 * nCount) / sqrt(double(nBlobsCount)) );


	// Cas ou on dispose des infos de localisation, on traite paquet par paquet les vertices
	if( paVtxLocalisation != 0 )
	{
		const LocalisationInfoArray& aVtxLocs = *paVtxLocalisation;

		for( int32 i = 0; i < nBlobsCount; ++i)
		{
			const BlobBase* pCurBlob = m_apBlobs[i];
			const BoundAabb& boxCurBlob   = pCurBlob->GetBlobBounds();

			VertexIndexArray& anVerticesIdx = influenced[i];
			anVerticesIdx.resize(0);
			anVerticesIdx.reserve(nEvalInfluencedPerBlob);


			typedef LocalisationInfoArray::const_iterator LocalConstIter;
			for( LocalConstIter itLoc = aVtxLocs.begin(); itLoc != aVtxLocs.end(); ++itLoc )
			{
				// Infos sur le bloc de vertices courant
				const ScanLineRange& scanVtxIndices = itLoc->first;

				int32 nFirstVtxIdx = std::max( scanVtxIndices.nRangeBeg, nStartIdx);
				int32 nEndVtxIdx   = std::min( scanVtxIndices.nRangeEnd, nStartIdx + nCount);
				int32 nVtxCount    = nEndVtxIdx - nFirstVtxIdx;

				if( nVtxCount > 0 && boxCurBlob.Intersects( itLoc->second ) )
				{
					pCurBlob->GradientAddAtSubArrayFillInfluenced( avNorms, aPos, anVerticesIdx, nFirstVtxIdx, nVtxCount);
				}
			}

		}
	}
	else
	{
		// On evalue la somme des gradients de la fonction en chaque position
		for( int32 i = 0; i < nBlobsCount; ++i)
		{
			const BlobBase*   pCurBlob = m_apBlobs[i];
			VertexIndexArray& anVerticesIdx = influenced[i];
			anVerticesIdx.resize(0);
			anVerticesIdx.reserve(nEvalInfluencedPerBlob);
			pCurBlob->GradientAddAtSubArrayFillInfluenced(avNorms, aPos, anVerticesIdx, nStartIdx, nCount);
		}
	}
}

                           /////////////////////////////


void IsoSurfaceBlobsObject::ComputeSurfaceNormalsAndInfluencedVtx( Vect3fArray& avNorms, BlobsInfluencesIndices& infl,
	                                                              const Vect3DArray& aPos,
	                                                              const LocalisationInfoArray* paVtxLocalisation /* = 0 */) const
{
	int32 nVtxCount = aPos.size();
	avNorms.resize( nVtxCount );
	ComputeSurfaceNormalsAndInfluencedVtxSubArray( avNorms, infl, aPos, 0, nVtxCount,paVtxLocalisation);
}


                           /////////////////////////////


void IsoSurfaceBlobsObject::ComputeSurfaceNormalsAndInfluencedVtxSubArray( Vect3fArray& avNorms, BlobsInfluencesIndices& infl,
	                                                                     const Vect3DArray& aPos,
	                                                                     int32 nFirstIndex, int32 nNormalsToCompute,
	                                                                     const LocalisationInfoArray* paVtxLocalisation /* = 0 */) const
{
	int32 nEndNormals = nFirstIndex + nNormalsToCompute;

	avNorms.resize( nEndNormals );
	
	// On met d'abord a 0 les normales
	std::fill( avNorms.begin()+nFirstIndex, avNorms.end(), Vector3f(0.f,0.f,0.f) );
//	for( int32 i = nFirstIndex; i < nEndNormals; ++i)
//		avNorms[i].Set(0.f, 0.f, 0.f);

	AddSurfaceNormalsFindInfluencedSubArray( avNorms, infl, aPos, nFirstIndex, nNormalsToCompute, paVtxLocalisation );
}



                           /////////////////////////////



void IsoSurfaceBlobsObject::ComputeSurfaceNormalsSubArray( Vect3fArray& avNorms, const Vect3DArray& aPos,
	                                                     int32 nFirstIndex, int32 nNormalsToCompute) const
{
	BlobsInfluencesIndices blobsInfluences;
	ComputeSurfaceNormalsAndInfluencedVtxSubArray( avNorms, blobsInfluences, aPos, nFirstIndex, nNormalsToCompute);
}



                           /////////////////////////////


bool IsoSurfaceBlobsObject::FindInterestingPartsOfPlane( AllScansArray& aScansPlane,
												const Vector3D& vcStart, const Vector3D& vcSteps,
												int32 nStepsX, int32 nStepsY) const
{
	// Decommenter la ligne suivante pour simplement remplir aScansPlane
//	return parent::FindInterestingPartsOfPlane( aScansPlane,vcStart, vcSteps,nStepsX, nStepsY);
	
	const Vector3D& vcMin = vcStart; // - Vector3D( 0.f, 0.f, vcSteps.z * .0001f);
	const Vector3D  vcMax = vcStart + Vector3D( vcSteps.x * float(nStepsX), vcSteps.y * float(nStepsY), vcSteps.z * 1.0001f);

	// On reinitialise les anciennes configurations...
	aScansPlane.ReInit();

	bool bAnythingInLayer = false;

	LengthType rDivStepX = 1.f/ vcSteps.x;
	LengthType rDivStepY = 1.f/ vcSteps.y;

	int32 nBlobsCount = m_apBlobs.size();
	for( int32 i = 0; i < nBlobsCount; ++i)
	{
		const BlobBase& curBlob = *(m_apBlobs[i]);

		Vector3D vcIntersMin, vcIntersMax;
		bool bIntersects = curBlob.ComputeMinEnclosingBoxAligned(vcIntersMin, vcIntersMax, vcMin, vcMax);
		if( bIntersects)
		{
			// On a la boite d'intersection entre la couche et les boites englobantes
			// On cherche maintenant les coordonnees entieres des cubes dans la couche
			int32 nBegX = FloorFast((vcIntersMin.x - vcStart.x) * rDivStepX);
			int32 nBegY = FloorFast((vcIntersMin.y - vcStart.y) * rDivStepY);

			int32 nEndX = CeilFast((vcIntersMax.x - vcStart.x) * rDivStepX);
			int32 nEndY = CeilFast((vcIntersMax.y - vcStart.y) * rDivStepY);

			if( nEndX >= nStepsX ) nEndX = nStepsX - 1;
			if( nEndY >= nStepsY ) nEndY = nStepsY - 1;

			assert( nBegX >=0 && nBegX < nStepsX);
			assert( nBegY >=0 && nBegY < nStepsY);
			assert( nEndX >=0 && nEndX < nStepsX);
			assert( nEndY >=0 && nEndY < nStepsY);

			ScanLineRange rangeX(nBegX, nEndX);
			ScanLineRange rangeY(nBegY, nEndY);
			aScansPlane.AddRectangle( rangeX, rangeY);

			bAnythingInLayer = true;
		}
	}
	return bAnythingInLayer;
}

//////////////////////////////////////////////////////////////////////////////

// Calcul de la boite englobante
// Retourne true si la boite englobante est bornee dans les 3 directions

bool IsoSurfaceBlobsObject::ComputeBounds( Vector3D& vcBoundMin, Vector3D& vcBoundMax ) const
{
	int32 nBlobsCount = m_apBlobs.size();
	
	// Si aucun blob, boite englobante de taille 0
	if( nBlobsCount == 0)
	{
		vcBoundMin = Vector3D(0,0,0);
		vcBoundMax = vcBoundMin;
		return true;
	}
	
	// Union des boites englobantes des blobs
	BoundAabb boundsUnion;
	
	bool bAllVisitedBlobsBounded = m_apBlobs[0]->GetBlobBounds(boundsUnion);

	for( int32 i = 1; bAllVisitedBlobsBounded && i < nBlobsCount; ++i)
	{
		const BlobBase* pCurBlob = m_apBlobs[i];
		BoundAabb  boundsCurBlob;
		bAllVisitedBlobsBounded = pCurBlob->GetBlobBounds(boundsCurBlob);
		boundsUnion.Merge( boundsCurBlob);
	}
	vcBoundMin = boundsUnion.GetMin();
	vcBoundMax = boundsUnion.GetMax();
	return bAllVisitedBlobsBounded;
}

                           /////////////////////////////

void IsoSurfaceBlobsObject::FindBlobsInfluences( BlobsInfluencesIndices& infl, const Vect3DArray& aVtxBlobs,
                                                 const LocalisationInfoArray* paVtxLocalisation /* = 0 */) const
{
	int32 nVtxCount = aVtxBlobs.size();
	FindBlobsInfluencesSubArray( infl, aVtxBlobs, 0, nVtxCount, paVtxLocalisation);
}
                           /////////////////////////////

void IsoSurfaceBlobsObject::FindBlobsInfluencesSubArray( BlobsInfluencesIndices& infl, const Vect3DArray& aVtxBlobs,
	                                                   int32 nVertexStart, int32 nVertexCount,
	                                                   const LocalisationInfoArray* paVtxLocalisation /* = 0 */) const
{
	int32 nBlobsCount = m_apBlobs.size();
	int32 nEvalVerticesPerBlob = (nVertexCount * 2) / nBlobsCount;

	infl.resize( nBlobsCount );

	// Si il n'y a qu'un blob, on est sur que tous les vertices sont influences par lui
	if( nBlobsCount == 1 )
	{
		VertexIndexArray& anCurBlobVtxIndices = infl[0];
		anCurBlobVtxIndices.resize( nVertexCount );
		for( int32 i = 0; i < nVertexCount; ++i)
			anCurBlobVtxIndices[i] = (nVertexStart + i);
	}
	else
	{
		// Cas ou on dispose des infos de localisation, on traite paquet par paquet les vertices
		if( paVtxLocalisation != 0 )
		{
			const LocalisationInfoArray& aVtxLocs = *paVtxLocalisation;

			for( int32 i = 0; i < nBlobsCount; ++i)
			{
				const BlobBase* pCurBlob = m_apBlobs[i];
				const BoundAabb& boxCurBlob = pCurBlob->GetBlobBounds();
	
				VertexIndexArray& anCurBlobVtxIndices = infl[i];
				anCurBlobVtxIndices.resize(0);
				anCurBlobVtxIndices.reserve(nEvalVerticesPerBlob);

				typedef LocalisationInfoArray::const_iterator LocalConstIter;
				for( LocalConstIter itLoc = aVtxLocs.begin(); itLoc != aVtxLocs.end(); ++itLoc )
				{
					// Infos sur le bloc de vertices courant
					const ScanLineRange& scanVtxIndices = itLoc->first;
					int32 nFirstVtxIdx = std::max( scanVtxIndices.nRangeBeg, nVertexStart);
					int32 nEndVtxIdx   = std::min( scanVtxIndices.nRangeEnd, nVertexStart + nVertexCount);
					int32 nVtxCount    = nEndVtxIdx - nFirstVtxIdx;

					if( nVtxCount > 0 && boxCurBlob.Intersects( itLoc->second ) )
					{
						pCurBlob->FindInfluencedVerticesSubArray( anCurBlobVtxIndices, aVtxBlobs, nFirstVtxIdx, nVtxCount);
					}
				}
			}
		}
		else
		{
			for( int32 i = 0; i < nBlobsCount; ++i)
			{
				const BlobBase* pCurBlob = m_apBlobs[i];
				VertexIndexArray& anCurBlobVtxIndices = infl[i];
				anCurBlobVtxIndices.resize(0);
				anCurBlobVtxIndices.reserve(nEvalVerticesPerBlob);
				pCurBlob->FindInfluencedVerticesSubArray( anCurBlobVtxIndices, aVtxBlobs, nVertexStart, nVertexCount);
			}
		}
	}
}

                           /////////////////////////////


void IsoSurfaceBlobsObject::ComputeBlobsInfluences( BlobsInfluencesValues& vals,
                                                    const BlobsInfluencesIndices& inflIdx, const Vect3DArray& aVtxBlobs ) const
{
	const int32 nBlobsCount = m_apBlobs.size();
	const int32 nVtxCount = aVtxBlobs.size();
	
	assert( inflIdx.size() == (std::size_t)nBlobsCount );
	vals.resize( nBlobsCount );

	if( nBlobsCount == 1)
	{
		VertexInfluenceArray&   afVals = vals[0];
		const VertexIndexArray& anIdx  = inflIdx[0];
		
		int32 nInfluenced = anIdx.size();

//		afVals.resize( nInfluenced );
		afVals.assign( nInfluenced, 1.f);
		return;
	}

	VertexInfluenceArray afTotalEachVtx(nVtxCount, 0.f);

	
	for( int32 i = 0; i < nBlobsCount; ++i)
	{
		const BlobBase* pCurBlob = m_apBlobs[i];
		const VertexIndexArray& anCurBlobVtxIndices = inflIdx[i];
		VertexInfluenceArray&   afCurBlobVtxValues  = vals[i];
		
		pCurBlob->EnergyGetAtInfluencedVertices( afCurBlobVtxValues, aVtxBlobs, anCurBlobVtxIndices);
	
		// Calcul du total pour chaque vertex
		int32 nInflVtxCount = anCurBlobVtxIndices.size();
		for( int32 j = 0; j < nInflVtxCount; ++j)
		{
			const int32 nCurIdx = anCurBlobVtxIndices[j];
			float& fCurInfl = afCurBlobVtxValues[j];
			
			// Les influences devront etre entre 0 et 1 (un blob negatif a une influence positive)
			fCurInfl = fabsf(fCurInfl);
			//fCurInfl = Sqrt(fCurInfl);
			afTotalEachVtx[nCurIdx] += fCurInfl;
		}
	}

	// Normalisation
	for( int32 i = 0; i < nBlobsCount; ++i)
	{
		const VertexIndexArray& anCurBlobVtxIndices = inflIdx[i];
		VertexInfluenceArray&   afCurBlobVtxValues  = vals[i];
		
		int32 nInflVtxCount = anCurBlobVtxIndices.size();
		for( int32 j = 0; j < nInflVtxCount; ++j)
		{
			const int32 nCurIdx = anCurBlobVtxIndices[j];
			afCurBlobVtxValues[j] /= afTotalEachVtx[nCurIdx];
		}
	}
}


//////////////////////////////////////////////////////////////////////////////


void IsoSurfaceBlobsObject::GenerateMesh( Triangulator& tessellator, 
                                   Vect3DArray& aVertices, TriangleArray& aTriangles,
                                   BlobsInfluencesIndices& blobsInfluences,
                                   Vect3fArray* paNorms /* = 0 */, bool bNormalizeNormals /* = false */) const
{
	aVertices.resize(0);
	aTriangles.resize(0);
	if( paNorms != 0)
		paNorms->resize(0);

	int32 nBlobsCount = m_apBlobs.size();
	//blobsInfluences.SetSize( 0 );
	blobsInfluences.resize( nBlobsCount );


	if( nBlobsCount == 0)
		return;

	if( nBlobsCount == 1)
	{
//		tessellator.GenerateMesh(*this, aVertices, aTriangles, paNorms, bNormalizeNormals);
		tessellator.GenerateMesh(*this, aVertices, aTriangles);

		if( paNorms != 0)
			ComputeSurfaceNormalsAndInfluencedVtx( *paNorms, blobsInfluences, aVertices);
		else
			FindBlobsInfluences( blobsInfluences, aVertices);
	}
	else
	{
		SBlobsDirectNeighborhood infosBlobsDirectNeighbors;
#if 0
		infosBlobsDirectNeighbors.ComputeDirectNeighborsAll( m_apBlobs);
#else
		// Memes resultats que ComputeDirectNeighborsAll mais plus rapide
		infosBlobsDirectNeighbors.ComputeDirectNeighborsSuccessors( m_apBlobs);
		infosBlobsDirectNeighbors.MakeDirectNeighborsSymetric();
#endif
		// Rassemblement des blobs par grappes (clusters)
		SBlobsCluster infosBlobsClusters( infosBlobsDirectNeighbors );

		// Creation d'une sous-isosurface pour chaque cluster.
		IsoSurfaceBlobsObject isoSurfaceCurCluster = *this;
		
		// Infos de localisation des vertices
		LocalisationInfoArray aLocInfo;
		aLocInfo.reserve(m_apBlobs.size() * 32);
		LocalisationInfoArray* pLocInfo;


		// Infos d'influences pour la sous-isosurface
		BlobsInfluencesIndices aBlobsInflIdxThisIso;

		// Creation de la geometrie cluster par cluster
		int32 nClustersCount = infosBlobsClusters.GetClustersCount();
		for(int32 i = 0; i < nClustersCount; ++i)
		{
			isoSurfaceCurCluster.RemoveAllBlobs();

			int32 nBlobsInCurCluster = infosBlobsClusters.GetBlobsCountInCluster( i );
			for(int32 j = 0; j < nBlobsInCurCluster; ++j)
			{
				int32 nBlobIdx = infosBlobsClusters.GetBlobIdxInCluster( i, j );
				isoSurfaceCurCluster.AddBlob( *(m_apBlobs[nBlobIdx]) );
			}
		
			int32 nCurIndexVtxStart = aVertices.size();

			// Creation de la geometrie uniquement (pas de normales)
			tessellator.GenerateMesh(isoSurfaceCurCluster, aVertices, aTriangles);


			Vect3DArray::const_iterator itNewVtxBeg = aVertices.begin() + nCurIndexVtxStart;
			Vect3DArray::const_iterator itNewVtxEnd = aVertices.end();

			int32 nCurVtxGenerated = itNewVtxEnd - itNewVtxBeg;

			// On ne s'interesse a la localisation des vertices si ca vaut le coup :
			//    -> seulement si il y a beaucoup de vertices et de blobs
//			if( nCurVtxGenerated > 10000 && nBlobsInCurCluster > 10 )
			if( nBlobsInCurCluster > 10 )
			{
				pLocInfo = &aLocInfo;
				pLocInfo->resize(0);
				BuildLocalisationInfo( *pLocInfo, itNewVtxBeg, itNewVtxEnd, nCurIndexVtxStart );
			}
			else
				pLocInfo = 0;

			aBlobsInflIdxThisIso.resize(0);
			// Identification des influences de chaque blob de la sous-isosurface sur les vertices generes
			if( paNorms != 0)
				isoSurfaceCurCluster.ComputeSurfaceNormalsAndInfluencedVtxSubArray( *paNorms, aBlobsInflIdxThisIso, aVertices, nCurIndexVtxStart, nCurVtxGenerated, pLocInfo);
			else
				isoSurfaceCurCluster.FindBlobsInfluencesSubArray( aBlobsInflIdxThisIso, aVertices, nCurIndexVtxStart, nCurVtxGenerated, pLocInfo);

			for( int32 j = 0; j < nBlobsInCurCluster; ++j)
			{
				int32 nBlobIdx = infosBlobsClusters.GetBlobIdxInCluster( i, j );

				VertexIndexArray& curBlobInflInfoInSubIso   = aBlobsInflIdxThisIso[j];
				VertexIndexArray& curBlobInflInfoInTotalIso = blobsInfluences[nBlobIdx];

				curBlobInflInfoInSubIso.swap( curBlobInflInfoInTotalIso );
				curBlobInflInfoInSubIso.resize(0);
			}
		}
	}

	// Normalisation des normales si cela a ete demande
	if( paNorms != 0 && bNormalizeNormals )
	{
//		std::for_each( paNorms->begin(), paNorms->end(), std::mem_fun(&Vector3f::Normalize) );
		int32 nVtxCount = aVertices.size();
		for( int32 i = 0; i < nVtxCount; ++i)
		{
			Vector3f& curNorm = (*paNorms)[i];
			curNorm.Normalize();
		}
	}
}


void IsoSurfaceBlobsObject::BuildLocalisationInfo( LocalisationInfoArray& aLocInfo,
                                                   const Vect3DArray::const_iterator itVtxBeg,
                                                   const Vect3DArray::const_iterator itVtxEnd,
                                                   int32 nFirstVtxIdx ) const
{
	if( itVtxBeg == itVtxEnd )
		return;

	// recuperation de la boite englobante de tous ces vertices
	// afin de limiter la taille des blocs de vertices

	BoundAabb englobVtices( *itVtxBeg, *itVtxBeg );

	typedef Vect3DArray::const_iterator VtxIter;
	for( Vect3DArray::const_iterator itVtx = itVtxBeg+1; itVtx != itVtxEnd; ++itVtx)
		englobVtices.UpdateForVertexAdd( *itVtx );

	Vector3D vcMaxBlockHalfSize = englobVtices.GetHalfSize() * ( 1.f / m_apBlobs.size() );

	// Calcul des blocs : ajout des vertices les uns apres les autres jusqu'a ce que la boite soit
	// trop grosse, auquel cas on entame un nouveau bloc

	ScanLineRange scanVtxBlock( nFirstVtxIdx, nFirstVtxIdx );
	BoundAabb          boxVtxBlock( *itVtxBeg, *itVtxBeg);

	int32 nCurVtxIdx = nFirstVtxIdx+1;
	for( Vect3DArray::const_iterator itVtx = itVtxBeg+1; itVtx != itVtxEnd; ++itVtx, ++nCurVtxIdx)
	{
		BoundAabb boxNext = boxVtxBlock;
		boxNext.UpdateForVertexAdd( *itVtx );
		
		const Vector3D vcBoxNewHalfSize = boxNext.GetHalfSize();
		// Si la nouvelle boite n'est pas trop grosse, on inclut le vertex dans le bloc
//		if( true )
		if(    vcBoxNewHalfSize.x < vcMaxBlockHalfSize.x
		    || vcBoxNewHalfSize.y < vcMaxBlockHalfSize.y
		    || vcBoxNewHalfSize.z < vcMaxBlockHalfSize.z )
		{
			boxVtxBlock = boxNext;
		}
		else
		{	
			// Sinon on termine le bloc, on l'enregistre...
			scanVtxBlock.nRangeEnd = nCurVtxIdx;
			aLocInfo.push_back( LocalisationInfo( scanVtxBlock, boxVtxBlock ) );

			// ...et on en commence un autre
			scanVtxBlock.nRangeBeg = nCurVtxIdx;
			scanVtxBlock.nRangeEnd = nCurVtxIdx;
			boxVtxBlock.SetMinMax( *itVtx, *itVtx );
		}
	}

	// Enregistrement du dernier bloc si ce n'est deja fait
	if( scanVtxBlock.nRangeEnd != nCurVtxIdx )
	{
		scanVtxBlock.nRangeEnd = nCurVtxIdx;
		aLocInfo.push_back( LocalisationInfo( scanVtxBlock, boxVtxBlock ) );
	}

	assert( nCurVtxIdx == (itVtxEnd - itVtxBeg + nFirstVtxIdx) );

#ifdef _DEBUG
	// Verification que tous les vertices initiaux sont representes
	typedef LocalisationInfoArray::const_iterator LocalConstIter;
	for( LocalConstIter itLoc = aLocInfo.begin()+1; itLoc != aLocInfo.end(); ++itLoc )
	{
		assert( (itLoc-1)->first.nRangeEnd == itLoc->first.nRangeBeg );
		for( int32 i = itLoc->first.nRangeBeg; i < itLoc->first.nRangeEnd; ++i)
		{
//			assert( itLoc->second.IsPointInside( *(itVtxBeg+i) ) );
		}
	}
	assert( aLocInfo.front().first.nRangeBeg == (nFirstVtxIdx));
	assert( aLocInfo.back().first.nRangeEnd  == (itVtxEnd - itVtxBeg + nFirstVtxIdx));
#endif

}




//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

void IsoSurfaceBlobsObject::SBlobsDirectNeighborhood::ComputeDirectNeighborsSuccessors( const BlobPtrArray& apBlobs )
{
	int32 nBlobsCount = apBlobs.size();
	int32 nEstimateNeighborsCount = nBlobsCount * nBlobsCount / (2 * 3);
	
	m_aDirectNeighborsInfos.resize(nBlobsCount);
	m_anDirectNeighborsIndices.resize(0);
	m_anDirectNeighborsIndices.reserve(nEstimateNeighborsCount);

	for( int32 j = 0; j < nBlobsCount; ++j)
	{
		SBlobDirectNeighborhoodInfo& curBlobNeighborhood = m_aDirectNeighborsInfos[j];
		curBlobNeighborhood.nBlobNeighborsFirstIndex = m_anDirectNeighborsIndices.size();
		curBlobNeighborhood.nBlobNeighborsCount = 0;

		const BlobBase* pCurBlob = apBlobs[j];
		assert( pCurBlob != 0);
		const BoundAabb& boundsCurBlob = pCurBlob->GetBlobBounds();

		// Recherche des voisins successeurs
		for( int32 i = j+1; i < nBlobsCount; ++i)
		{
			const BlobBase* pCurBlobNeighb = apBlobs[i];
			assert( pCurBlobNeighb != 0);
			const BoundAabb& boundsCurNeighbor = pCurBlobNeighb->GetBlobBounds();
			if( boundsCurBlob.Intersects( boundsCurNeighbor) )
				m_anDirectNeighborsIndices.push_back(i);
		}
		int32 nCurNeighborsSize = m_anDirectNeighborsIndices.size();
		int32 nFirstCurNeighborhood = curBlobNeighborhood.nBlobNeighborsFirstIndex;
		curBlobNeighborhood.nBlobNeighborsCount = nCurNeighborsSize - nFirstCurNeighborhood;
	}
}

//////////////////////////////////////////////////////////////////////////////


void IsoSurfaceBlobsObject::SBlobsDirectNeighborhood::ComputeDirectNeighborsAll( const BlobPtrArray& apBlobs )
{
	int32 nBlobsCount = apBlobs.size();
	m_aDirectNeighborsInfos.resize(nBlobsCount);
	m_anDirectNeighborsIndices.resize(0);
	m_anDirectNeighborsIndices.reserve(nBlobsCount);

	for( int32 j = 0; j < nBlobsCount; ++j)
	{
		SBlobDirectNeighborhoodInfo& curBlobNeighborhood = m_aDirectNeighborsInfos[j];
		curBlobNeighborhood.nBlobNeighborsFirstIndex = m_anDirectNeighborsIndices.size();
		curBlobNeighborhood.nBlobNeighborsCount = 0;

		const BlobBase* pCurBlob = apBlobs[j];
		assert( pCurBlob != 0);
		const BoundAabb& boundsCurBlob = pCurBlob->GetBlobBounds();

		for( int32 i = 0; i < nBlobsCount; ++i)
		{
			if( i != j)
			{
				const BlobBase* pCurBlobNeighb = apBlobs[i];
				assert( pCurBlobNeighb != 0);
				const BoundAabb& boundsCurNeighbor = pCurBlobNeighb->GetBlobBounds();
				if( boundsCurBlob.Intersects( boundsCurNeighbor) )
					m_anDirectNeighborsIndices.push_back(i);
			}
		}
		int32 nCurNeighborsSize = m_anDirectNeighborsIndices.size();
		int32 nFirstCurNeighborhood = curBlobNeighborhood.nBlobNeighborsFirstIndex;
		curBlobNeighborhood.nBlobNeighborsCount = nCurNeighborsSize - nFirstCurNeighborhood;
	}
}


//////////////////////////////////////////////////////////////////////////////


void IsoSurfaceBlobsObject::SBlobsDirectNeighborhood::MakeDirectNeighborsSymetric()
{
	std::vector<int32>&               anBlobsNeighborsIdx = m_anDirectNeighborsIndices; 
	SBlobDirectNeighborhoodInfoArray& aBlobNeighborsInfo  = m_aDirectNeighborsInfos;

	int32 nBlobsCount = aBlobNeighborsInfo.size();

	// Il faut rendre la description du voisinage symetrique :
	// Si blob1 a blob2 dans sa liste de voisins, 
	// il faut aussi que blob2 ait blob1 dans la sienne.

	int32 nAsymNeighborsCount = anBlobsNeighborsIdx.size();

	// On cree un tableau intermediaire contenant pour chaque blob son nombre total de voisins
	
	// Etape 1 : on le rempli avec le nombre de voisins "asymetrique" : voisins qui sont apres dans la liste
	std::vector<int32> anBlobsNeighborsCounts;
	anBlobsNeighborsCounts.resize( nBlobsCount );
	
	for( int32 i = 0; i < nBlobsCount; ++i)
		anBlobsNeighborsCounts[i] = aBlobNeighborsInfo[i].nBlobNeighborsCount;

	// Etape 2 : Pour chaque blob, on informe ses voisins successeurs de 
	//           l'existence d'un voisin supplementaire a compter
	for( int32 j = 0; j < nBlobsCount; ++j)
	{
		const SBlobDirectNeighborhoodInfo& curBlobNeighborhood = aBlobNeighborsInfo[j];
		int32 nNeighborsFirst = curBlobNeighborhood.nBlobNeighborsFirstIndex;
		int32 nNeighborsLast  = nNeighborsFirst + curBlobNeighborhood.nBlobNeighborsCount;
		for( int32 i = nNeighborsFirst; i < nNeighborsLast; ++i)
		{
			int32 nNeighborIndex = anBlobsNeighborsIdx[i];
			++ anBlobsNeighborsCounts[nNeighborIndex];
		}
	}

	// Preparation du tableau descriptif symetrique
	// On commence par l'initialiser a partir de l'original.
	// A partir du tableau des compteurs precemment calcule, 
	// on deduit l'index du premier index de voisin pour chaque blob
	SBlobDirectNeighborhoodInfoArray aBlobTotalNeighborsInfo; 
	aBlobTotalNeighborsInfo.reserve( nBlobsCount );

	int32 nCurFirstIdx = 0;
	for( int32 i = 0; i < nBlobsCount; ++i)
	{
		int32 nCountCur = anBlobsNeighborsCounts[i];
		 // First + Count = ou placer le prochain index (ici, prochain == premier)

		SBlobDirectNeighborhoodInfo newInfo;

		newInfo.nBlobNeighborsCount = 0;
		newInfo.nBlobNeighborsFirstIndex = nCurFirstIdx;
		nCurFirstIdx += nCountCur;

		aBlobTotalNeighborsInfo.push_back( newInfo );
	}
	
	// Maintenant, chaque blob copie ses successeurs de la description de depart
	// et renseigne chaque successeur sur sa presence en tant que predecesseur

	// la symetrie va doubler le voisinage
	int32 nNeighboursCountTotal = nAsymNeighborsCount*2;
	std::vector<int32> anBlobsNeighborsIdxSym(nNeighboursCountTotal);

	for( int32 i = 0; i < nBlobsCount; ++i)
	{
		const SBlobDirectNeighborhoodInfo& curNeighborAsymInfo = aBlobNeighborsInfo[i];

		int32 nAsymIdxCount = curNeighborAsymInfo.nBlobNeighborsCount;

		if( nAsymIdxCount > 0)
		{
			int32 nAsymStartIdx = curNeighborAsymInfo.nBlobNeighborsFirstIndex;
			int32 nAsymEndIdx   = nAsymStartIdx + nAsymIdxCount;
			
			// Recuperation des successeurs calcules d'origine
			int32& nCountTotalNeighbours1 = aBlobTotalNeighborsInfo[i].nBlobNeighborsCount;
			int32  nFirstTotalNeighbours1 = aBlobTotalNeighborsInfo[i].nBlobNeighborsFirstIndex;
		
			std::vector<int32>::const_iterator itCopySrcBeg = anBlobsNeighborsIdx.begin() + nAsymStartIdx;
			std::vector<int32>::const_iterator itCopySrcEnd = itCopySrcBeg + nAsymIdxCount;
			std::vector<int32>::iterator       itCopyDest   = anBlobsNeighborsIdxSym.begin() + nFirstTotalNeighbours1 + nCountTotalNeighbours1;
			
//			std::copy( itCopySrcBeg, itCopySrcEnd, itCopyDest );
			for( std::vector<int32>::const_iterator it = itCopySrcBeg; it != itCopySrcEnd; ++it)
				*itCopyDest++ = *it;
				       
//			memcpy( anBlobsNeighborsIdxSym.GetBuffer()+nFirstTotalNeighbours1 + nCountTotalNeighbours1,
//				    anBlobsNeighborsIdx.GetBuffer()+nAsymStartIdx,
//					nAsymIdxCount * sizeof(int32));

			nCountTotalNeighbours1 += nAsymIdxCount;

			// Predecesseurs
			for( int j = nAsymStartIdx; j < nAsymEndIdx; ++j)
			{
				// i est voisin de nIdxAsymNeighb
				int32 nIdxAsymNeighb = anBlobsNeighborsIdx[j]; 
				int32& nCountTotalNeighbours2 = aBlobTotalNeighborsInfo[nIdxAsymNeighb].nBlobNeighborsCount;
				int32  nFirstTotalNeighbours2 = aBlobTotalNeighborsInfo[nIdxAsymNeighb].nBlobNeighborsFirstIndex;
				anBlobsNeighborsIdxSym[nFirstTotalNeighbours2 + nCountTotalNeighbours2] = i;
				++nCountTotalNeighbours2;
			}
		}
	}
	aBlobTotalNeighborsInfo.swap(aBlobNeighborsInfo);
	anBlobsNeighborsIdxSym.swap(anBlobsNeighborsIdx);
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

IsoSurfaceBlobsObject::SBlobsCluster::SBlobsCluster(const SBlobsDirectNeighborhood& neighbInfo)
: m_anBlobsClusterId(neighbInfo.GetBlobsCount())
{
	BuildFromNeighborhood(neighbInfo);
}

//////////////////////////////////////////////////////////////////////////////

void IsoSurfaceBlobsObject::SBlobsCluster::BuildFromNeighborhood( const SBlobsDirectNeighborhood& neighInfo)
{
	int32 nBlobsCount = neighInfo.GetBlobsCount();

	m_anBlobsIdSortedByCluster.reserve(nBlobsCount);
	m_anBlobsClusterId.resize(nBlobsCount);
	
	m_anClustersFirstIdx.reserve(nBlobsCount/8);
	m_anClustersFirstIdx.resize(0);

	std::stack<int32> stackClusterIdsToVisit;
//	stackClusterIdsToVisit.reserve(nBlobsCount); // Estimation : au pire on visite tout
//	stackClusterIdsToVisit.EnsureCapacity(nBlobsCount); // Estimation : au pire on visite tout
	
	for( int32 nCurBlob = 0; nCurBlob < nBlobsCount; ++nCurBlob)
		m_anBlobsClusterId[nCurBlob] = nCurBlob;

	for( int32 nCurBlob = 0; nCurBlob < nBlobsCount; ++nCurBlob)
	{
		// On ne travaille sur ce blob que si il n'a pas ete rattache a un cluster
		bool bIsBlobToProcess = (m_anBlobsClusterId[nCurBlob] == nCurBlob);
		if( bIsBlobToProcess )
		{
			// Nouveau cluster : on stocke le premier indice de ce cluster dans la liste triee
			int32 nFirstClusterIdx = m_anBlobsIdSortedByCluster.size();
			m_anClustersFirstIdx.push_back(nFirstClusterIdx);

			assert( stackClusterIdsToVisit.empty() );
			stackClusterIdsToVisit.push(nCurBlob);

			while( ! stackClusterIdsToVisit.empty() )
			{
				int32 nBlobToProcessIdx   = stackClusterIdsToVisit.top();
				stackClusterIdsToVisit.pop();
				m_anBlobsIdSortedByCluster.push_back( nBlobToProcessIdx);

				SBlobsDirectNeighborhood::NeighborIdxConstIter itCurNeighb, itEndNeighb;
				neighInfo.GetStartEndNeighborConstIters( itCurNeighb, itEndNeighb, nBlobToProcessIdx);

				for( ; itCurNeighb != itEndNeighb; ++itCurNeighb)
				{
					int32 nCurSecondNeighbor = *itCurNeighb;
				
					bool bNeighborAlreadyInCluster = (m_anBlobsClusterId[nCurSecondNeighbor] == nCurBlob);
					if(  ! bNeighborAlreadyInCluster )
					{
						assert(m_anBlobsClusterId[nCurSecondNeighbor] == nCurSecondNeighbor);
						m_anBlobsClusterId[nCurSecondNeighbor] = nCurBlob;
						stackClusterIdsToVisit.push(nCurSecondNeighbor);
					}
				}
			}
		}
	}
}



//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////





} // namespace
